#include <BDK.h>

#include <math.h>
#define ROFFSET 1
#define GOFFSET 0
#define BOFFSET 2

#define RUN_AVG_W1 1.0f
#define RUN_AVG_W2 10.0f

#define TOUCH_PIN_TOP A1
#define TOUCH_PIN_MID A8
#define TOUCH_PIN_BOT A9

#define ONE_G 16384.0f

BDK::BDK():
  strip(Adafruit_NeoPixel(led_count, 17, NEO_GRB + NEO_KHZ800))
{
  uint32_t *val_id_h = (uint32_t*)0x40048058;
  uint32_t *val_id_m = (uint32_t*)0x4004805C;
  uint32_t *val_id_l = (uint32_t*)0x40048060;

  float ktavg = 0.0f;
  float kmavg = 0.0f;
  float kbavg = 0.0f;

  for (int c = 0; c < 40; c++) {
    ktavg = (RUN_AVG_W2 * ktavg + RUN_AVG_W1 * touchRead(TOUCH_PIN_TOP)) / (RUN_AVG_W2 + RUN_AVG_W1);
    kmavg = (RUN_AVG_W2 * kmavg + RUN_AVG_W1 * touchRead(TOUCH_PIN_MID)) / (RUN_AVG_W2 + RUN_AVG_W1);
    kbavg = (RUN_AVG_W2 * kbavg + RUN_AVG_W1 * touchRead(TOUCH_PIN_BOT)) / (RUN_AVG_W2 + RUN_AVG_W1);
    delay(5);
  }

  key_top_baseline = ktavg;
  key_middle_baseline = kmavg;
  key_bottom_baseline = kbavg;

  init_screen();
  init_mpu6050();
  init_leds();
}

void BDK::init_screen() {
  // by default, we'll generate the high voltage from the 3.3v line internally! (neat!)
  display.begin(SSD1306_SWITCHCAPVCC, 0x3C);  // initialize with the I2C addr 0x3D (for the 128x64)
  // init done

  display.clearDisplay();

  display.setTextSize(1);
  display.setTextColor(WHITE);
}

void BDK::init_leds() {
  delay(100);
  strip.begin();
  for (uint8_t i = 0; i < led_count; i++) {
    strip.setPixelColor(i, 0);
  }
  strip.show();
}

void BDK::init_mpu6050() {
  accelgyro.initialize();
}

void BDK::setPixelRGBA(uint16_t c, uint8_t r, uint8_t g, uint8_t b, uint8_t in_a) {
  if (c < 0 || c > led_count) {
    return;
  }
  uint16_t a = in_a;
  uint8_t *pixels = strip.getPixels();
  if (a != 255) {
    uint8_t dest_r = pixels[c * 3 + ROFFSET];
    uint8_t dest_g = pixels[c * 3 + GOFFSET];
    uint8_t dest_b = pixels[c * 3 + BOFFSET];

    uint16_t acc_r = (dest_r * (255 - a)) + (r * a);
    uint16_t acc_g = (dest_g * (255 - a)) + (g * a);
    uint16_t acc_b = (dest_b * (255 - a)) + (b * a);

    pixels[c * 3 + ROFFSET] = (uint8_t)(acc_r >> 8 & 0xFF);
    pixels[c * 3 + GOFFSET] = (uint8_t)(acc_g >> 8 & 0xFF);
    pixels[c * 3 + BOFFSET] = (uint8_t)(acc_b >> 8 & 0xFF);
  } else {
    pixels[c * 3 + ROFFSET] = r;
    pixels[c * 3 + GOFFSET] = g;
    pixels[c * 3 + BOFFSET] = b;
  }
}

void BDK::setPixelRGBASub(uint16_t c, uint8_t in_s, uint8_t r, uint8_t g, uint8_t b, uint8_t in_a) {
  if (c < 0 || c > led_count) {
    return;
  }

  if (in_s < 5) {
      setPixelRGBA((c - 1 + led_count) % led_count, r, g, b, in_a);
      return;
  }
  if (in_s > 250) {
      setPixelRGBA(c, r, g, b, in_a);
      return;
  }

  uint16_t s = in_s;

  uint8_t c_a = (s * in_a) >> 8;
  uint8_t n_a = ((255 - s) * in_a) >> 8;
  setPixelRGBA(c, r, g, b, c_a);

  // TODO: Need to configure wrapping behavior
  setPixelRGBA((c - 1 + led_count) % led_count, r, g, b, n_a);
}

void BDK::setPixel(float p, uint8_t r, uint8_t g, uint8_t b) {
  setPixel(p, r, g, b, 255);
}
void BDK::setPixel(float p, uint8_t r, uint8_t g, uint8_t b, uint8_t a) {
  uint16_t ip = floor(p);
  setPixelRGBASub(ip, (p - ip) * 256, r, g, b, a);
}

void BDK::setPixelHue(float p, uint16_t h, uint8_t s, uint8_t v) {
  setPixelHue(p, h, s, v, 255);
}

void BDK::setPixelHue(float p, uint16_t hue, uint8_t sat, uint8_t val, uint8_t a) {
  const uint8_t dim_curve[] = {
    0,   1,   1,   2,   2,   2,   2,   2,   2,   3,   3,   3,   3,   3,   3,   3,
    3,   3,   3,   3,   3,   3,   3,   4,   4,   4,   4,   4,   4,   4,   4,   4,
    4,   4,   4,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   6,   6,   6,
    6,   6,   6,   6,   6,   7,   7,   7,   7,   7,   7,   7,   8,   8,   8,   8,
    8,   8,   9,   9,   9,   9,   9,   9,   10,  10,  10,  10,  10,  11,  11,  11,
    11,  11,  12,  12,  12,  12,  12,  13,  13,  13,  13,  14,  14,  14,  14,  15,
    15,  15,  16,  16,  16,  16,  17,  17,  17,  18,  18,  18,  19,  19,  19,  20,
    20,  20,  21,  21,  22,  22,  22,  23,  23,  24,  24,  25,  25,  25,  26,  26,
    27,  27,  28,  28,  29,  29,  30,  30,  31,  32,  32,  33,  33,  34,  35,  35,
    36,  36,  37,  38,  38,  39,  40,  40,  41,  42,  43,  43,  44,  45,  46,  47,
    48,  48,  49,  50,  51,  52,  53,  54,  55,  56,  57,  58,  59,  60,  61,  62,
    63,  64,  65,  66,  68,  69,  70,  71,  73,  74,  75,  76,  78,  79,  81,  82,
    83,  85,  86,  88,  90,  91,  93,  94,  96,  98,  99,  101, 103, 105, 107, 109,
    110, 112, 114, 116, 118, 121, 123, 125, 127, 129, 132, 134, 136, 139, 141, 144,
    146, 149, 151, 154, 157, 159, 162, 165, 168, 171, 174, 177, 180, 183, 186, 190,
    193, 196, 200, 203, 207, 211, 214, 218, 222, 226, 230, 234, 238, 242, 248, 255,
  };


  val = dim_curve[val];
  sat = 255-dim_curve[255-sat];

  uint8_t r = 0;
  uint8_t g = 0;
  uint8_t b = 0;
  uint8_t base;

  if (sat == 0) { // Fast case for unsaturated colors
    r = val;
    g = val;
    b = val;
  } else {

    base = ((255 - sat) * val)>>8;

    switch(hue/60) {
        case 0:
                r = val;
                g = (((val-base)*hue)/60)+base;
                b = base;
        break;

        case 1:
                r = (((val-base)*(60-(hue%60)))/60)+base;
                g = val;
                b = base;
        break;

        case 2:
                r = base;
                g = val;
                b = (((val-base)*(hue%60))/60)+base;
        break;

        case 3:
                r = base;
                g = (((val-base)*(60-(hue%60)))/60)+base;
                b = val;
        break;

        case 4:
                r = (((val-base)*(hue%60))/60)+base;
                g = base;
                b = val;
        break;

        case 5:
                r = val;
                g = base;
                b = (((val-base)*(60-(hue%60)))/60)+base;
        break;
    }
  }
  setPixel(p, r, g, b, a);
}


void BDK::updateLEDs() {
  strip.show();
}
void BDK::clearLEDs() {
  for (uint8_t i = 0; i < led_count; i++) {
    strip.setPixelColor(i, 0);
  }
}

void BDK::fadeLEDs(uint8_t in_a) {
  uint16_t a = in_a;
  uint8_t *pixels = strip.getPixels();
  for (uint8_t i = 0; i < led_count * 3; i++) {
    pixels[i] = (uint8_t)((pixels[i] * in_a) >> 8);
  }
}


bool BDK::touchTop() {
  return touchRead(TOUCH_PIN_TOP) > 500 + key_top_baseline;
}

bool BDK::touchMiddle() {
  return touchRead(TOUCH_PIN_MID) > 500 + key_middle_baseline;
}

bool BDK::touchBottom() {
  return touchRead(TOUCH_PIN_BOT) > 500 + key_bottom_baseline;
}

void BDK::updateAccelerometer() {

  int16_t ax, ay, az;
  int16_t gx, gy, gz;

  accelgyro.getMotion6(&ax, &ay, &az, &gx, &gy, &gz);

  accel.x = ax / ONE_G;
  accel.y = ay / ONE_G;
  accel.z = az / ONE_G;

  gyro.x = gx;
  gyro.y = gy;
  gyro.z = gz;

  accel_intensity = sqrt((double)ax * ax + ay * ay + az * az);
  rotation = atan2(ax, ay);
  pitch = atan2(ax, az);
}
